% This script generates Fig. 11.

% We plot the average best quotes in the last episode for different values
% of sigma and for three states:

% in period tau = 1
% in period tau = 2, conditionally on a trade in tau = 1
% in period tau = 2, conditionally on no trade in tau = 1

% Additionally, the figure plots the value of the best quote in each case 
% in the theoretical benchmark (Nash equilibrium).

%Pre-allocate a 9*3 matrix with 9 rows corresponding to 9 values of sigma
%and 3 columns with the averages of a_min_1, a_min_2_NT, a_min_2_T.

mat_av = zeros(9,3);

%Pre-allocate a 9*3 matrix with 9 rows corresponding to 9 values of sigma
%and 3 columns with 1.96 times standard deviations/sqrt(observations) of a_min_1, a_min_2_NT, a_min_2_T.

mat_sd = zeros(9,3);

%Fill the matrix by loading the data on last episodes for every sigma.

for sigma = 1:9
% Create the name of the dataset to be loaded
    dataname = sprintf('Data/last_episodes_%d.txt', sigma);
    % Load the data
    last_episodes = readmatrix(dataname);
    %Compute a_min_1.
    a_min_1 = min([last_episodes(:,1) last_episodes(:,3)],[],2);
    %Compute a_min_2_NT. First we keep only episodes with no trade in tau=1.
    index = find(last_episodes(:,15)>0); %Column 15 is equal to 1 if there was no trade in tau=1.
    notrades = last_episodes(index,:);
    %Then we compute a_min_2 over these episodes. The period 2 prices are in
    %columns 2 (for AMM 1) and 4 (for AMM 2).
    a_min_2_NT = min([notrades(:,2) notrades(:,4)],[],2);
    %Compute a_min_2_T. First we keep only episodes with trade in tau=1.
    index = find(last_episodes(:,15)==0); %Column 15 is equal to 0 if there was a trade.
    trades = last_episodes(index,:);
    %Then we compute a_min_2 over these episodes. The period 2 prices are in
    %columns 2 (for AMM 1) and 4 (for AMM 2).
    a_min_2_T = min([trades(:,2) trades(:,4)],[],2);
    %Fill the mat_av matrix with the averages of a_min_1, a_min_2_NT, a_min_2_T.
    mat_av(sigma,1) = mean(a_min_1);
    mat_av(sigma,2) = mean(a_min_2_NT);
    mat_av(sigma,3) = mean(a_min_2_T);
    %Fill the mat_sd matrix with 1.96* std. deviation of each price, divided by the
    %square root of the number of observations.
    [n1,~] = size(last_episodes);
    [n2,~] = size(notrades);
    [n3,~] = size(trades);
    mat_sd(sigma,1) = 1.96*std(a_min_1)/sqrt(n1);
    mat_sd(sigma,2) = 1.96*std(a_min_2_NT)/sqrt(n2);
    mat_sd(sigma,3) = 1.96*std(a_min_2_T)/sqrt(n3);
end

%We create a matrix with the theoretical values for each sigma of a_min_1,
%a_min_2_NT and a_min_2_NT. The indexing is the same as in "mat".

mat_theory = zeros(9,3);

%Parameters
N = 2;
vH=4;
vL=0;
grid_middle = 8;
grid_size= 69;  
tick= 0.1;      

for sigma=1:9

%We start with a_min_1.
%We compute the theoretical expected profit associated with price a_1.
%We have Pi(a) = Pr(v=vH) Pr(vH + l > a) (a-vH) + Pr(v=vL) Pr(vL + l > a) (a-vL)
%We compute Pi(a) for every price in the grid. 

%NE is a matrix with 2*grid_size+1 lines (all prices on the grid), first
%column is the price, second column the associated profit, third column a
%variable equal to 1 if a is a NE price.

NE = zeros(2*grid_size+1,3);

%Compute the profit for each price.
for i=1:2*grid_size+1
    a = (grid_middle - grid_size*tick - tick)+(i*tick);
    NE(i,1) = a;
    profit = 0.5*(1-normcdf(a-vH,0,sigma))*(a-vH) + 0.5*(1-normcdf(a-vL,0,sigma))*(a-vL);
    NE(i,2) = profit;
end

%Check for each price whether it can be a NE, using Lemma 3 in the Online
%Appendix. We start with the second price, all prices below 2.0 can never be a NE
%price anyway.

%We set NE(:,3) to 1. We will replace with a zero each price for which we
%find a profitable deviation (not quoting anything or undercutting).
NE(:,3) = ones(2*grid_size+1,1);
%We know the first price necessarily generates losses.
NE(1,3) = 0;
for i=2:2*grid_size+1
%Check that the profit is positive.    
if NE(i,2) < 0
   NE(i,3) = 0;
end

%Check that having 1/N times the profit is better than the full profit at a lower price.    
for j=1:i-1
    if NE(j,2) > (1/N)*NE(i,2)
    NE(i,3) = 0;
    end
end

end

%Find highest NE price.
index = find(NE(:,3)==1,1,'last');
a1 = (grid_middle - grid_size*tick - tick)+(index*tick);
%Record the a_min_1:
mat_theory(sigma,1) = a1;

%We now compute a_min_2_NT.
%The method is exactly the same as for a_min_1, except that the probability
%that v = vH is no longer 1/2 but the Bayesian probability given the
%observation of no trade in tau = 1.

%We start by computing the posterior probability mu that v = vH.
%Note that the period 1 ask is still a1, computed on line 118.
mu = 0.5*normcdf(a1-vH,0,sigma)/(0.5*normcdf(a1-vH,0,sigma)+0.5*normcdf(a1-vL,0,sigma));

%NE is a matrix with 2*grid_size+1 lines (all prices on the grid), first
%column is the price, second column the associated profit, third column a
%variable equal to 1 if a is a NE price.

NE = zeros(2*grid_size+1,3);

%Compute the profit for each price.
for i=1:2*grid_size+1
    a = (grid_middle - grid_size*tick - tick)+(i*tick);
    NE(i,1) = a;
    profit = mu*(1-normcdf(a-vH,0,sigma))*(a-vH) + (1-mu)*(1-normcdf(a-vL,0,sigma))*(a-vL);
    NE(i,2) = profit;
end

%Check for each price whether it can be a NE, using Lemma 3 in the Online
%Appendix. We start with the second price, all prices below 2.0 can never be a NE
%price anyway.

%We set NE(:,3) to 1. We will replace with a zero each price for which we
%find a profitable deviation (not quoting anything or undercutting).
NE(:,3) = ones(2*grid_size+1,1);
%We know the first price necessarily generates losses.
NE(1,3) = 0;
for i=2:2*grid_size+1
%Check that the profit is positive.    
if NE(i,2) < 0
   NE(i,3) = 0;
end

%Check that having 1/N times the profit is better than the full profit at a lower price.    
for j=1:i-1
    if NE(j,2) > (1/N)*NE(i,2)
    NE(i,3) = 0;
    end
end

end

%Find highest NE price.
index = find(NE(:,3)==1,1,'last');
a2_NT = (grid_middle - grid_size*tick - tick)+(index*tick);
%Record the a_min_2_NT:
mat_theory(sigma,2) = a2_NT;


%We finally compute a_min_2_T.
%The method is exactly the same as for a_min_1, except that the probability
%that v = vH is no longer 1/2 but the Bayesian probability given the
%observation of a trade in tau = 1.

%We start by computing the posterior probability mu that v = vH.
%Note that the period 1 ask is still a, computed on line 118.
mu = 0.5*(1-normcdf(a1-vH,0,sigma))/(0.5*(1-normcdf(a1-vH,0,sigma))+0.5*(1-normcdf(a1-vL,0,sigma)));

%NE is a matrix with 2*grid_size+1 lines (all prices on the grid), first
%column is the price, second column the associated profit, third column a
%variable equal to 1 if a is a NE price.

NE = zeros(2*grid_size+1,3);

%Compute the profit for each price.
for i=1:2*grid_size+1
    a = (grid_middle - grid_size*tick - tick)+(i*tick);
    NE(i,1) = a;
    profit = mu*(1-normcdf(a-vH,0,sigma))*(a-vH) + (1-mu)*(1-normcdf(a-vL,0,sigma))*(a-vL);
    NE(i,2) = profit;
end

%Check for each price whether it can be a NE, using Lemma 3 in the Online
%Appendix. We start with the second price, all prices below 2.0 can never be a NE
%price anyway.

%We set NE(:,3) to 1. We will replace with a zero each price for which we
%find a profitable deviation (not quoting anything or undercutting).
NE(:,3) = ones(2*grid_size+1,1);
%We know the first price necessarily generates losses.
NE(1,3) = 0;
for i=2:2*grid_size+1
%Check that the profit is positive.    
if NE(i,2) < 0
   NE(i,3) = 0;
end

%Check that having 1/N times the profit is better than the full profit at a lower price.    
for j=1:i-1
    if NE(j,2) > (1/N)*NE(i,2)
    NE(i,3) = 0;
    end
end

end

%Find highest NE price.
index = find(NE(:,3)==1,1,'last');
a2_T = (grid_middle - grid_size*tick - tick)+(index*tick);
%Record the a_min_2_T:
mat_theory(sigma,3) = a2_T;

end

%% MEAN FINAL BEST QUOTES
figure();
hold on;

%Averages, a_min_1
plot(1:9, mat_av(:,1), 'Color',[0.0,0.0,1.0], 'LineWidth', 1);
%Confidence Interval
ci_a1 = fill([1:9, flip(1:9)], [(mat_av(:,1)-mat_sd(:,1))', flip((mat_av(:,1)+mat_sd(:,1))')], 'b');
set(ci_a1, 'facealpha', 0.3);  % make the confidence interval transparent
%Averages, a_min_2_NT
plot(1:9, mat_av(:,2), 'Color',[0.0,1.0,0.0], 'LineWidth', 1);
%Confidence Interval
ci_a2NT = fill([1:9, flip(1:9)], [(mat_av(:,2)-mat_sd(:,2))', flip((mat_av(:,2)+mat_sd(:,2))')], 'g');
set(ci_a2NT, 'facealpha', 0.3);  % make the confidence interval transparent
%Averages, a_min_2_T
plot(1:9, mat_av(:,3), 'Color',[1.0,0.0,0.0], 'LineWidth', 1);
%Confidence Interval
ci_a2T = fill([1:9, flip(1:9)], [(mat_av(:,3)-mat_sd(:,3))', flip((mat_av(:,3)+mat_sd(:,3))')], 'r');
set(ci_a2T, 'facealpha', 0.3);  % make the confidence interval transparent

%Theoretical Benchmark
plot(1:9,mat_theory(:,1) ,'--o','Color',[0.0,0.0,1.0]) %a_min_1
plot(1:9,mat_theory(:,2) ,'--o','Color',[0.0,1.0,0.0]) %a_min_2_NT
plot(1:9,mat_theory(:,3) ,'--o','Color',[1.0,0.0,0.0]) %a_min_2_T

ylabel("Average Best Quote");
xlabel('$\sigma$', 'Interpreter', 'latex', 'FontSize', 14); 

legend('Simulations - $\bar{a}_1$', '95\% Confidence interval', 'Simulations - $\bar{a}_2^{NT}$', '95\% Confidence interval' , 'Simulations - $\bar{a}_2^{T}$', '95\% Confidence interval','Nash Equilibrium - $\bar{a}_1$','Nash Equilibrium - $\bar{a}_2^{NT}$', 'Nash Equilibrium - $\bar{a}_2^{T}$','Location','Northwest','Interpreter','Latex','Fontsize',8);
legend('Box','off');
legend('NumColumns',1);
ylim([0,10]);
hold off;

%save png
filename = sprintf('/Figures/Fig_11.png');
saveas(gcf, [pwd filename]);

